/************* COPYRIGHT AND CONFIDENTIALITY INFORMATION ********************
**                                                                         **
** Copyright (c) 2010 technicolor                                          **
** All Rights Reserved                                                     **
**                                                                         **
** This program contains proprietary information which is a trade          **
** secret of technicolor and/or its affiliates and also is protected as    **
** an unpublished work under applicable Copyright laws. Recipient is       **
** to retain this program in confidence and is not permitted to use or     **
** make copies thereof other than as permitted in a written agreement      **
** with technicolor, UNLESS OTHERWISE EXPRESSLY ALLOWED BY APPLICABLE LAWS.**
**                                                                         **
****************************************************************************/

/** \file
 * Multimedia Switch kernel module.
 *
 * \version v1.0
 *
 *************************************************************************/


/*
 * Define tracing prefix, needs to be defined before includes.
 */
#define MODULE_NAME    "MMSWITCH"
/*########################################################################
#                                                                        #
#  HEADER (INCLUDE) SECTION                                              #
#                                                                        #
########################################################################*/
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <net/sock.h>
#include <linux/socket.h>
#include <linux/file.h>
#include <linux/udp.h>
#include <net/ip.h>
#include <linux/jiffies.h>
#include <linux/kfifo.h>
#include <linux/spinlock.h>
#include <linux/sched.h>
#include <linux/kthread.h>

#include "mmswitch_netlink_p.h"
#include "mmconnuser_netlink_p.h"
#include "mmconn_p.h"
#include "mmconn_netlink_p.h"
#include "mmconnuser_p.h"
#include "mmconnkernel_p.h"
#include "mmconnmulticast_p.h"
#include "mmconntone_p.h"
#include "mmconnrelay_p.h"
#include "mmconnrtcp_p.h"

/*########################################################################
#                                                                        #
#  MACROS/DEFINES                                                        #
#                                                                        #
########################################################################*/

/*########################################################################
#                                                                        #
#  TYPES                                                                 #
#                                                                        #
########################################################################*/

/*########################################################################
#                                                                        #
#  PRIVATE GLOBALS                                                       #
#                                                                        #
########################################################################*/
static int _traceLevel    = MMPBX_TRACELEVEL_ERROR;
static struct task_struct *_writeThread  = NULL;
static struct semaphore   _writeSem;
static LIST_HEAD(_writeQueue); /**< Queue for all data which we want to write to a socket */
static spinlock_t _listReadLock;
static struct semaphore _exitReadSem;

static struct task_struct *_readThread = NULL;
static struct semaphore   _readSem;
static LIST_HEAD(_readQueue); /**< Queue for all data which we want to write to a socket */
static spinlock_t _listWriteLock;
static struct semaphore _exitWriteSem;

static int _exit = 0;


/*########################################################################
#                                                                        #
#  PRIVATE FUNCTION PROTOTYPES                                           #
#                                                                        #
########################################################################*/
static void write_space_socket(struct sock *sk);
static void change_state_socket(struct sock *sk);
static void data_ready_socket(struct sock *sk,
                              int         bytes);
static int mmSwitchWriteSocketThreadFunc(void *thread_data);
static int mmSwitchReadSocketThreadFunc(void *thread_data);


static void timer_cb(unsigned long data);

/*########################################################################
#                                                                        #
#   PUBLIC FUNCTION DEFINITIONS                                          #
#                                                                        #
########################################################################*/

/**
 *
 */
MmPbxError mmSwitchSetTraceLevel(MmPbxTraceLevel level)
{
    _traceLevel = level;
    MMPBX_TRC_INFO("New trace level : %s\n", mmPbxGetTraceLevelString(level));


    return MMPBX_ERROR_NOERROR;
}

/**
 *
 */
MmPbxError mmSwitchPrepareSocket(MmSwitchSock *mmSwitchSock,
                                 int          sockFd)
{
    MmPbxError  ret  = MMPBX_ERROR_NOERROR;
    int         err  = 0;


    do {
        MMPBX_TRC_INFO("sockFd: %d\n", sockFd);
        if (sockFd >= 0) {
            /* Get struct socket from file descriptor created in user space */
            mmSwitchSock->sock = sockfd_lookup(sockFd, &err);
            if (!mmSwitchSock->sock) {
                MMPBX_TRC_ERROR("sockfd_lookup failed with: %d", err);
                ret = MMPBX_ERROR_INTERNALERROR;
                break;
            }

            /* Replace some of the internal socket members */
            mmSwitchSock->sock->sk->sk_state_change  = change_state_socket;
            mmSwitchSock->sock->sk->sk_data_ready    = data_ready_socket;
            mmSwitchSock->sock->sk->sk_write_space   = write_space_socket;
            mmSwitchSock->sock->sk->sk_user_data     = (void *)mmSwitchSock;

            /* Need to set the socket allocation to atomic */
            mmSwitchSock->sock->sk->sk_allocation = GFP_ATOMIC;

            if (mmSwitchSock->sock->file != NULL) {
                /* Needed to decrement refcount of socket, otherwise close in userspace will not close the socket */
                sockfd_put(mmSwitchSock->sock);
            }
        }

        return err;
    } while (0);

    return ret;
}

/**
 *
 */
MmPbxError mmSwitchCleanupSocket(MmSwitchSock *mmSwitchSock)
{
    MmPbxError              ret = MMPBX_ERROR_NOERROR;
    unsigned long           flags;
    MmSwitchSockQueueEntry  *current_entry = NULL;
    MmSwitchSockQueueEntry  *next_entry    = NULL;

    mmSwitchSock->cb     = NULL;
    mmSwitchSock->mmConn = NULL;

    /* Remove references to this socket from the read and write queues */
    spin_lock_irqsave(&_listWriteLock, flags);
    list_for_each_entry_safe(current_entry, next_entry, &_writeQueue, list)
    {
        if (current_entry->mmSwitchSock == mmSwitchSock) {
            list_del(&current_entry->list);
            kfree(current_entry);
        }
    }
    spin_unlock_irqrestore(&_listWriteLock, flags);

    spin_lock_irqsave(&_listReadLock, flags);
    list_for_each_entry_safe(current_entry, next_entry, &_readQueue, list)
    {
        if (current_entry->mmSwitchSock == mmSwitchSock) {
            list_del(&current_entry->list);
            kfree(current_entry);
        }
    }
    spin_unlock_irqrestore(&_listReadLock, flags);

    return ret;
}

/**
 *
 */
MmPbxError mmSwitchRegisterSocketCb(MmSwitchSock        *mmSwitchSock,
                                    MmSwitchSocketCb    cb,
                                    MmConnHndl          conn,
                                    MmConnPacketHeader  *header)
{
    MmPbxError ret = MMPBX_ERROR_NOERROR;

    mmSwitchSock->cb     = cb;
    mmSwitchSock->mmConn = conn;
    memcpy(&mmSwitchSock->header, header, sizeof(MmConnPacketHeader));

    return ret;
}

/**
 *
 */
MmPbxError mmSwitchWriteSocket(MmSwitchSock     *mmSwitchSock,
                               struct sockaddr  *dst,
                               void             *data,
                               unsigned int     len)
{
    MmPbxError              err              = MMPBX_ERROR_NOERROR;
    MmSwitchSockQueueEntry  *sockQueueEntry  = NULL;
    unsigned long           flags;

    /*
     * We will send data to socket asynchronously.
     * This is done to avoid "BUG scheduling while atomic", triggered within sock_sendmsg.
     * This means that it's possible that this function is called multiple times before
     * the data is realy send, that's why we queue the data.
     */

    if (len > MAX_MMSWITCHSOCK_DATA_SIZE) {
        err = MMPBX_ERROR_NORESOURCES;
        MMPBX_TRC_ERROR("len > MAX_MMSWITCHSOCK_DATA_SIZE\n");
        return err;
    }

    /*
     * Check whether the socket is still in use.
     */
    if (mmSwitchSock->cb != NULL) {
        /* Try to allocate another sockQueueEntry object instance */
        sockQueueEntry = kmalloc(sizeof(MmSwitchSockQueueEntry), GFP_ATOMIC);
        if (sockQueueEntry == NULL) {
            err = MMPBX_ERROR_NORESOURCES;
            MMPBX_TRC_ERROR("%s\n", mmPbxGetErrorString(err));
            return err;
        }

        /* Add reference to mmswitchSock */
        sockQueueEntry->mmSwitchSock = mmSwitchSock;

        /* copy data */
        memcpy(&sockQueueEntry->dst, dst, sizeof(struct sockaddr_storage));
        sockQueueEntry->datalen = len;
        memcpy(sockQueueEntry->data, data, len);

        /*Add sockQueueEntry to write queue */
        INIT_LIST_HEAD(&sockQueueEntry->list);
        spin_lock_irqsave(&_listWriteLock, flags);
        list_add(&sockQueueEntry->list, &_writeQueue);
        spin_unlock_irqrestore(&_listWriteLock, flags);

        /*Release semaphore of socket write thread */
        if (!_exit) {
            up(&_writeSem);
        }
    }

    return err;
}

/**
 *
 */
MmPbxError mmSwitchPrepareTimer(MmSwitchTimer   *tmr,
                                MmSwitchTimerCb cb,
                                MmConnHndl      mmConn,
                                int             periodic)
{
    MmPbxError ret = MMPBX_ERROR_NOERROR;

    if (tmr == NULL) {
        ret = MMPBX_ERROR_INVALIDPARAMS;
        MMPBX_TRC_INFO("%s\r\n", mmPbxGetErrorString(ret));
        return ret;
    }

    if (cb == NULL) {
        ret = MMPBX_ERROR_INVALIDPARAMS;
        MMPBX_TRC_INFO("%s\r\n", mmPbxGetErrorString(ret));
        return ret;
    }

    if (mmConn == NULL) {
        ret = MMPBX_ERROR_INVALIDPARAMS;
        MMPBX_TRC_INFO("%s\r\n", mmPbxGetErrorString(ret));
        return ret;
    }

    if (tmr) {
        tmr->mmSwitchTimerCb = cb;
        tmr->mmConn          = mmConn;
        tmr->active          = TIMER_IDLE;
        tmr->periodic        = periodic;
        init_timer(&tmr->timer);
        tmr->timer.function  = timer_cb;
        tmr->timer.data      = (unsigned long)tmr;
    }

    return ret;
}

/**
 *
 */
MmPbxError mmSwitchDestroyTimer(MmSwitchTimer *tmr)
{
    MmPbxError ret = MMPBX_ERROR_NOERROR;

    if (tmr == NULL) {
        ret = MMPBX_ERROR_INVALIDPARAMS;
        MMPBX_TRC_INFO("%s\r\n", mmPbxGetErrorString(ret));
        return ret;
    }

    if (tmr->active == TIMER_ACTIVE) {
        tmr->active = TIMER_IDLE;
        del_timer_sync(&tmr->timer);
    }

    return ret;
}

/**
 *  Start timer, period in ms.
 */
MmPbxError mmSwitchStartTimer(MmSwitchTimer *tmr,
                              int           period)
{
    MmPbxError ret = MMPBX_ERROR_NOERROR;

    if (tmr == NULL) {
        ret = MMPBX_ERROR_INVALIDPARAMS;
        MMPBX_TRC_INFO("%s\r\n", mmPbxGetErrorString(ret));
        return ret;
    }

    if (period == 0) {
        ret = MMPBX_ERROR_INVALIDPARAMS;
        MMPBX_TRC_INFO("%s, period == 0\r\n", mmPbxGetErrorString(ret));
        return ret;
    }

    tmr->period        = period;
    tmr->timer.expires = jiffies + HZ * period / 1000;
    tmr->active        = TIMER_ACTIVE;
    add_timer(&tmr->timer);

    return ret;
}

/**
 *
 */
MmPbxError mmSwitchStopTimer(MmSwitchTimer *tmr)
{
    MmPbxError ret = MMPBX_ERROR_NOERROR;

    if (tmr == NULL) {
        ret = MMPBX_ERROR_INVALIDPARAMS;
        MMPBX_TRC_INFO("%s\r\n", mmPbxGetErrorString(ret));
        return ret;
    }

    if (tmr->active == TIMER_ACTIVE) {
        tmr->active = TIMER_IDLE;
        del_timer_sync(&tmr->timer);
    }

    return ret;
}

/**
 *
 */

MmPbxError mmSwitchRestartTimer(MmSwitchTimer *tmr,
                                int           period)
{
    MmPbxError ret = MMPBX_ERROR_NOERROR;

    do {
        ret = mmSwitchStopTimer(tmr);
        if (ret != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmSwitchStopTimer failed with: %s\n", mmPbxGetErrorString(ret));
            break;
        }

        ret = mmSwitchStartTimer(tmr, period);
        if (ret != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmSwitchStartTimer failed with: %s\n", mmPbxGetErrorString(ret));
            break;
        }


        return ret;
    } while (0);

    return ret;
}

/*########################################################################
#                                                                        #
#   PRIVATE FUNCTION DEFINITIONS                                         #
#                                                                        #
########################################################################*/

/**
 *
 */
static void timer_cb(unsigned long data)
{
    MmSwitchTimer *t = (MmSwitchTimer *)data;

    if (t->mmConn != NULL) {
        /* Call callback */
        t->mmSwitchTimerCb(t->mmConn);

        /* restart if periodic */
        if (t->periodic) {
            if (t->active == TIMER_ACTIVE) {
                t->timer.expires = jiffies + HZ * t->period / 1000;
                add_timer(&t->timer);
            }
        }
    }
}

/**
 *
 */
static void write_space_socket(struct sock *sk)
{
    /* Not implemented */
}

/**
 *
 */
static void change_state_socket(struct sock *sk)
{
    /* Not implemented */
}

/**
 *
 */
static void data_ready_socket(struct sock *sk,
                              int         bytes)
{
    MmSwitchSock            *mmSwitchSock    = (MmSwitchSock *) sk->sk_user_data;
    MmSwitchSockQueueEntry  *sockQueueEntry  = NULL;
    unsigned long           flags;

    if (mmSwitchSock->cb != NULL) {
        /* Try to allocate another sockQueueEntry object instance */
        sockQueueEntry = kmalloc(sizeof(MmSwitchSockQueueEntry), GFP_ATOMIC);
        if (sockQueueEntry == NULL) {
            MMPBX_TRC_ERROR("%s\n", "kmalloc failed\n");
            return;
        }

        /* copy data into entry*/
        sockQueueEntry->mmSwitchSock = mmSwitchSock;
        /*Add sockQueueEntry to read queue */
        INIT_LIST_HEAD(&sockQueueEntry->list);
        spin_lock_irqsave(&_listReadLock, flags);
        list_add(&sockQueueEntry->list, &_readQueue);
        spin_unlock_irqrestore(&_listReadLock, flags);

        /*Release semaphore of socket read thread */
        if (!_exit) {
            up(&_readSem);
        }
    }
}

static int mmSwitchWriteSocketThreadFunc(void *thread_data)
{
    int                     ret = 0;
    struct kvec             vec;
    struct msghdr           msg;
    MmSwitchSockQueueEntry  *sockQueueEntry = NULL;
    unsigned long           flags;


    while (1) {
        /* Attempt to acquire semaphore */
        ret = down_interruptible(&_writeSem);

        if (ret != 0) {
            MMPBX_TRC_DEBUG("signal received, semaphore not acquired...\n");
            continue;
        }

        while (1) {
            sockQueueEntry = NULL;
            spin_lock_irqsave(&_listWriteLock, flags);
            if (!list_empty(&_writeQueue)) {
                sockQueueEntry = list_first_entry(&_writeQueue, MmSwitchSockQueueEntry, list);
                list_del(&sockQueueEntry->list);
            }
            spin_unlock_irqrestore(&_listWriteLock, flags);

            if (!sockQueueEntry) {
                break;
            }
            if (_exit) {
                kfree(sockQueueEntry);
                break;
            }

            if (sockQueueEntry->mmSwitchSock->sock != NULL) {
                /* Do not forget these memsets or kernel_sendmsg will fail */
                memset(&vec, 0, sizeof(struct kvec));
                memset(&msg, 0, sizeof(struct msghdr));

                vec.iov_base     = sockQueueEntry->data;
                vec.iov_len      = sockQueueEntry->datalen;
                msg.msg_flags    = MSG_DONTWAIT | MSG_NOSIGNAL;
                msg.msg_name     = &sockQueueEntry->dst;
                msg.msg_namelen  = (sockQueueEntry->dst.ss_family == AF_INET) ? sizeof(struct sockaddr_in) : sizeof(struct sockaddr_in6);

                ret = kernel_sendmsg(sockQueueEntry->mmSwitchSock->sock, &msg, &vec, 1, vec.iov_len);
                if (ret < 0) {
                    MMPBX_TRC_DEBUG("kernel_sendmsg failed with error: %d\n", ret);
                }
            }

            /*Destruct */
            kfree(sockQueueEntry);
        }
        if (_exit) {
            MMPBX_TRC_DEBUG("exited \n");
            break;
        }
    }

    up(&_exitWriteSem);

    do_exit(0);

    return 0;
}

static int mmSwitchReadSocketThreadFunc(void *thread_data)
{
    int                     ret = 0;
    struct kvec             vec;
    struct msghdr           msg;
    MmSwitchSockQueueEntry  *sockQueueEntry  = NULL;
    int                     size             = 0;
    unsigned long           flags;
    MmSwitchSocketCb        cb;
    MmConnHndl              mmConn;
 


    while (1) {
        /* Attempt to acquire semaphore */
        ret = down_interruptible(&_readSem);

        if (ret != 0) {
            MMPBX_TRC_DEBUG("signal received, semaphore not acquired...\n");
            continue;
        }

        while (1) {
            sockQueueEntry = NULL;
            spin_lock_irqsave(&_listReadLock, flags);
            if (!list_empty(&_readQueue)) {
                sockQueueEntry = list_first_entry(&_readQueue, MmSwitchSockQueueEntry, list);
                list_del(&sockQueueEntry->list);
            }
            spin_unlock_irqrestore(&_listReadLock, flags);

            if (!sockQueueEntry) {
                break;
            }
            if (_exit) {
                kfree(sockQueueEntry);
                break;
            }

            cb = sockQueueEntry->mmSwitchSock->cb;
            mmConn = sockQueueEntry->mmSwitchSock->mmConn ;

            if (cb != NULL) {
                memset(&vec, 0, sizeof(struct kvec));
                memset(&msg, 0, sizeof(struct msghdr));
                vec.iov_base = sockQueueEntry->data;
                vec.iov_len  = sizeof(sockQueueEntry->data);

                size = kernel_recvmsg(sockQueueEntry->mmSwitchSock->sock, &msg, &vec, 1, vec.iov_len, MSG_DONTWAIT);
                if (size > 0) {
                    ret = cb(mmConn, sockQueueEntry->data, &sockQueueEntry->mmSwitchSock->header, size);
                    if (ret != MMPBX_ERROR_NOERROR) {
                        MMPBX_TRC_ERROR("sockQueueEntry->mmSwitchSock->cb failed with error: %d\n", ret);
                    }
                }
            }

            /*Destruct */
            kfree(sockQueueEntry);
        }
        if (_exit) {
            MMPBX_TRC_DEBUG("exited\n");
            break;
        }
    }

    up(&_exitReadSem);

    do_exit(0);

    return 0;
}

/**
 *
 */
static int __init mmSwitchInit(void)
{
    MmPbxError  err  = MMPBX_ERROR_NOERROR;
    int         ret  = 0;

    do {
        spin_lock_init(&_listWriteLock);
        spin_lock_init(&_listReadLock);

        err = mmSwitchGeNlInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmSwitchGenNlInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }

        err = mmConnUsrGeNlInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmConnUsrGenNlInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }

        err = mmConnGeNlInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmConngeNlInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }
        err = mmConnInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmConnInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }

        err = mmConnUsrInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmConnUsrInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }

        err = mmConnKrnlInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmConnKrnlInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }

        err = mmConnMulticastInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmConnMulticastInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }

        err = mmConnToneInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmConnToneInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }

        err = mmConnRelayInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmConnRelayInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }

        err = mmConnRtcpInit();
        if (err != MMPBX_ERROR_NOERROR) {
            MMPBX_TRC_ERROR("mmConnRtcpInit failed with: %s\n", mmPbxGetErrorString(ret));
            ret = -1;
            break;
        }

        /*Create dedicated kernel threads for socket handling*/
        sema_init(&_writeSem, 0);
        _writeThread = kthread_run(mmSwitchWriteSocketThreadFunc, NULL, "mmswitchsock_w");

        if (_writeThread == NULL) {
            MMPBX_TRC_ERROR("kthread_run failed\n");
            BUG();
            break;
        }

        sema_init(&_readSem, 0);
        _readThread = kthread_run(mmSwitchReadSocketThreadFunc, NULL, "mmswitchsock_r");
        if (_readThread == NULL) {
            MMPBX_TRC_ERROR("kthread_run failed\n");
            BUG();
            break;
        }

        sema_init(&_exitReadSem, 0);

        sema_init(&_exitWriteSem, 0);

        return ret;
    } while (0);

    return ret;
}

/**
 *
 */
static void __exit mmSwitchExit(void)
{
    MmPbxError              err              = MMPBX_ERROR_NOERROR;
    MmSwitchSockQueueEntry  *sockQueueEntry  = NULL;
    struct list_head        *pos, *q;

    MMPBX_TRC_DEBUG("Start mmSwitchExit  \n");

    _exit = 1;

    up(&_readSem);
    while (1) {
        if (down_interruptible(&_exitReadSem) == 0) {
            break;
        }
    }

    up(&_writeSem);
    while (1) {
        if (down_interruptible(&_exitWriteSem) == 0) {
            break;
        }
    }

    MMPBX_TRC_DEBUG("Switch Read and Write processes are terminated\n");

    err = mmSwitchGeNlDeinit();
    if (err != MMPBX_ERROR_NOERROR) {
        MMPBX_TRC_ERROR("Failed to deinitialize mmswitch communication channel\n");
    }

    err = mmConnGeNlDeinit();
    if (err != MMPBX_ERROR_NOERROR) {
        MMPBX_TRC_ERROR("mmConngeNlDeInit failed with: %s\n", mmPbxGetErrorString(err));
    }

    err = mmConnUsrGeNlDeinit();
    if (err != MMPBX_ERROR_NOERROR) {
        MMPBX_TRC_ERROR("Failed to deinitialize mmConnUsr communication channel\n");
    }

    err = mmConnUsrDeinit();
    if (err != MMPBX_ERROR_NOERROR) {
        MMPBX_TRC_ERROR("Failed to deinitialize mmConnUsr channel\n");
    }

    err = mmConnKrnlDeinit();
    if (err != MMPBX_ERROR_NOERROR) {
        MMPBX_TRC_ERROR("mmConnKrnlDeinit failed with: %s\n", mmPbxGetErrorString(err));
    }

    err = mmConnMulticastDeinit();
    if (err != MMPBX_ERROR_NOERROR) {
        MMPBX_TRC_ERROR("mmConnMulticastDeInit failed with: %s\n", mmPbxGetErrorString(err));
    }

    err = mmConnToneDeinit();
    if (err != MMPBX_ERROR_NOERROR) {
        MMPBX_TRC_ERROR("mmConnToneDeinit failed with: %s\n", mmPbxGetErrorString(err));
    }

    err = mmConnRelayDeinit();
    if (err != MMPBX_ERROR_NOERROR) {
        MMPBX_TRC_ERROR("mmConnRelayDeinit failed with: %s\n", mmPbxGetErrorString(err));
    }

    err = mmConnRtcpDeinit();
    if (err != MMPBX_ERROR_NOERROR) {
        MMPBX_TRC_ERROR("mmConnRtcpDeinit failed with: %s\n", mmPbxGetErrorString(err));
    }

    list_for_each_safe(pos, q, &_writeQueue)
    {
        sockQueueEntry = list_entry(pos, MmSwitchSockQueueEntry, list);

        /*Remove from list with all MmSwitchSockQueueEntries structures */
        list_del(&sockQueueEntry->list);
        /*Destruct */
        kfree(sockQueueEntry);
    }

    list_for_each_safe(pos, q, &_readQueue)
    {
        sockQueueEntry = list_entry(pos, MmSwitchSockQueueEntry, list);

        /*Remove from list with all MmSwitchSockQueueEntries structures */
        list_del(&sockQueueEntry->list);
        /*Destruct */
        kfree(sockQueueEntry);
    }
    MMPBX_TRC_DEBUG("mmSwitchExit successfully\n");
}

/*GPL license is needed to create dicated workqueue */
MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("Jurgen Schoeters <jurgen.schoeters@technicolor.com>");
module_init(mmSwitchInit);
module_exit(mmSwitchExit);
